<?php

namespace App\Nova;

use App\Models\Role;
use App\Models\User;
use App\Nova\Filters\PermissionExtraValue;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Http\Request;
use Laravel\Nova\Actions\ActionResource;
use Laravel\Nova\Actions\ExportAsCsv;
use Laravel\Nova\Auth\PasswordValidationRules;
use Laravel\Nova\Fields\BelongsToMany;
use Laravel\Nova\Fields\Boolean;
use Laravel\Nova\Fields\Gravatar;
use Laravel\Nova\Fields\HasMany;
use Laravel\Nova\Fields\ID;
use Laravel\Nova\Fields\MorphToMany;
use Laravel\Nova\Fields\Password;
use Laravel\Nova\Fields\PasswordConfirmation;
use Laravel\Nova\Fields\Text;
use Laravel\Nova\Fields\URL;
use Laravel\Nova\Http\Requests\NovaRequest;
use Laravel\Nova\Nova;
use Laravel\Nova\Tabs\Tab;
use MZiraki\PersianDateField\PersianDateTime;

class UserResource extends Resource
{
    use PasswordValidationRules;

    /**
     * The model the resource corresponds to.
     *
     * @var class-string<\App\Models\User>
     */
    public static $model = \App\Models\User::class;

    /**
     * The single value that should be used to represent the resource when being displayed.
     *
     * @var string
     */
    public static $title = 'name';

    /**
     * The columns that should be searched.
     *
     * @var array
     */
    public static $search = [
        'name', 'email',
    ];


    public static $group = "Guard";

    /**
     * Get the fields displayed by the resource.
     *
     * @return array<int, \Laravel\Nova\Fields\Field|\Laravel\Nova\Panel|\Laravel\Nova\ResourceTool|\Illuminate\Http\Resources\MergeValue>
     */
    public function fields(NovaRequest $request): array
    {
        return [
            ID::make()->sortable()->onlyOnDetail(),


            Text::make(__('Name'), 'name')
                ->sortable()
                ->rules('required', 'max:255'),

            Gravatar::make()->maxWidth(50),

            Text::make(__('Email'), 'email')
                ->sortable()
                ->rules('required', 'email', 'max:254')
                ->creationRules('unique:users,email')
                ->updateRules('unique:users,email,{{resourceId}}'),

            Password::make(__('Password'), 'password')
                ->onlyOnForms()
                ->creationRules($this->passwordRules())
                ->updateRules($this->optionalPasswordRules())
                ->canSeeWhen('user.not-being-impersonated',$this->model()),

            PasswordConfirmation::make(__('Password Confirmation'),'password_confirmation')
                ->canSeeWhen('user.not-being-impersonated',$this->model()),

            PersianDateTime::make(__('Updated At'), 'updated_at')
                ->color('rgb('.config('nova.brand.colors.500').')') // customize color
                ->sortable()
                ->exceptOnForms()
                ->onlyOnDetail(),

            PersianDateTime::make(__('Created At'), 'created_at')
                ->color('rgb('.config('nova.brand.colors.500').')') // customize color
                ->sortable()
                ->exceptOnForms()
                ->onlyOnDetail(),


            Boolean::make(__('Is Active'), 'is_active')
                ->canSeeWhen('superAdmin',$this->model()),

            Tab::group('Relations', [
                MorphToMany::make(__('Roles'), 'roles', RoleResource::class)
                    ->canSee(function ($request) {
                        return $request->user()->hasPermissionTo('role.view-any');
                    }),
                MorphToMany::make(__('Permissions'), 'fixedPermissions', PermissionResource::class)
                    ->fields(function ($request, $relatedModel) {
                        $model = data_get($relatedModel,'model');
                        $url = null;
                        $label = '-';
                        if (filled($model) && class_exists('\\'.$model)) {
                            $resource = Nova::resourceForModel($model);
                            $label = $resource::singularLabel()." | ".data_get($relatedModel,'pivot.target_id') ;
                            $url = route('nova.pages.detail', ['resource' => $resource::uriKey(),'resourceId' => data_get($relatedModel,'pivot.target_id')]);
                        }
                        return [
                            Text::make(__('Extra'),'target_id'),
                            URL::make(__('Link'),'link', fn () => $url)
                                ->displayUsing(fn () => $label)->exceptOnForms(),

                        ];
                    })
                    ->relatableQueryUsing(function (NovaRequest $request, Builder $query) {
                        $ids = (User::query()->find($request->resourceId)->loadMissing('roles.permissions')->roles)
                            ->pluck('permissions')
                            ->flatten()
                            ->where('pivot.extra','*')
                            ->pluck('id');
                        return $query->whereNotIn('id',$ids);
                    })
                    ->allowDuplicateRelations()
                    ->canSee(function ($request) {
                        return $request->user()->hasPermissionTo('permission.view-any');
                    }),
            ]),


        ];
    }

    /**
     * Get the cards available for the request.
     *
     * @return array<int, \Laravel\Nova\Card>
     */
    public function cards(NovaRequest $request): array
    {
        return [];
    }

    /**
     * Get the filters available for the resource.
     *
     * @return array<int, \Laravel\Nova\Filters\Filter>
     */
    public function filters(NovaRequest $request): array
    {
        return [];
    }

    /**
     * Get the lenses available for the resource.
     *
     * @return array<int, \Laravel\Nova\Lenses\Lens>
     */
    public function lenses(NovaRequest $request): array
    {
        return [];
    }

    /**
     * Get the actions available for the resource.
     *
     * @return array<int, \Laravel\Nova\Actions\Action>
     */
    public function actions(NovaRequest $request): array
    {
        return [
            ExportAsCsv::make(),
        ];
    }

    public function menu(Request $request)
    {
        return parent::menu($request)->withBadge(function () {
            return static::$model::count();
        });
    }



//    public function authorizedToReplicate(Request $request)
//    {
//        return false;
//    }

/*    public function authorizedToImpersonate(NovaRequest $request)
    {
        return parent::authorizedToImpersonate($request) &&
            $request->user()->can('canImpersonate',$this->model()) &&
            $request->user()->hasPermissionTo('canImpersonateUser','web',$request->user());
    }*/
}
